using System;
using System.Windows.Forms;
using System.CodeDom;
using System.Collections;
using System.ComponentModel;
using System.Diagnostics;
using Borland.Studio.ToolsAPI;
using BeWise.SharpBuilderTools.Gui;
using BeWise.SharpBuilderTools.Utils;

namespace BeWise.SharpBuilderTools
{
	public class OTAClipboard : BaseOTA
	{
		private class ClipboardHook : NativeWindow, IDisposable {

			/**************************************************************/
			/*                         Constructor
			/**************************************************************/

			public void Create() {
				CreateParams _CP = new CreateParams();

				if (System.Environment.OSVersion.Version.Major >= 5)
					_CP.Parent = (IntPtr)HWND_MESSAGE;

				CreateHandle(_CP);
			}

			/**************************************************************/
			/*                       Destructor
			/**************************************************************/

			protected void Dispose(bool disposing) {
				if (disposing)
					DestroyHandle();
			}


			~ClipboardHook() {
				Dispose(false);
			}

			public void Dispose()
			{
				Dispose(true);
				GC.SuppressFinalize(this);
			}

			/**************************************************************/
			/*                       Private
			/**************************************************************/

			// Win32 functions Mapping
			private const int WM_CHANGECBCHAIN = 0x030D;
			private const int WM_DRAWCLIPBOARD = 0x0308;
			private const int HWND_MESSAGE = -3;

			private IntPtr _NextClipboardWindow;
			private EventHandler fChanged;

			/**************************************************************/
			/*                       Protected
			/**************************************************************/

			protected override void WndProc(ref Message m) {
				if (m.Msg == WM_CHANGECBCHAIN) {
					if (_NextClipboardWindow == m.WParam)
						_NextClipboardWindow = m.LParam;
					else
						WinAPI.SendMessage(_NextClipboardWindow, WM_CHANGECBCHAIN, m.WParam, m.LParam);
				}
				else if (m.Msg == WM_DRAWCLIPBOARD) {
					try {
						OnChanged();
					} finally {
						WinAPI.SendMessage(_NextClipboardWindow, WM_DRAWCLIPBOARD, m.WParam, m.LParam);
					}
				}
				else
					base.WndProc(ref m);
			}

			protected void OnChanged() {
				if (fChanged != null)
					fChanged(this, new EventArgs());
			}

			/**************************************************************/
			/*                       Public
			/**************************************************************/

			public override void CreateHandle(CreateParams aCp) {
				base.CreateHandle(aCp);

				_NextClipboardWindow = WinAPI.SetClipboardViewer(Handle);
			}

			public override void DestroyHandle() {
				WinAPI.ChangeClipboardChain(Handle, _NextClipboardWindow);
				base.DestroyHandle();
			}

			/**************************************************************/
			/*                       Events
			/**************************************************************/

			public event EventHandler Changed {
				add {	fChanged += value; }
				remove {	fChanged -= value; }
			}
		}

		/**************************************************************/
		/*                        Private
		/**************************************************************/

		private ArrayList fClipBoardList = new ArrayList();
		private ClipboardHook fClipboardHook;

		/**************************************************************/
		/*                        Protected
		/**************************************************************/

		protected void ClipBoardChanged(object aSender, EventArgs e) {
			IDataObject _Data = Clipboard.GetDataObject();

			if (_Data.GetDataPresent(DataFormats.Text))
			{
				fClipBoardList.Insert(0, _Data.GetData(DataFormats.Text).ToString());

				if (fClipBoardList.Count > Main.Configuration.ClipBoardStackSize) {
					fClipBoardList.RemoveAt(Main.Configuration.ClipBoardStackSize);
				}
			}
		}

		/**************************************************************/
		/*                        Public
		/**************************************************************/

		public void DoOpenClipboardViewer(object aSender, EventArgs AEventArgs) {
			FrmClipboardViewer _Frm = new FrmClipboardViewer();

			_Frm.Items = fClipBoardList;
			_Frm.ShowDialog();
		}

		public override void IDERegister(out string[] aMenuNames, out int[] aMenuShortCuts) {
			IOTAMainMenuService _MenuService = null;
			IOTAMenuItem _MenuItem = null;

			_MenuService = (IOTAMainMenuService) BorlandIDE.GetService(typeof(IOTAMainMenuService));

			// Clipboard
			_MenuItem = _MenuService.AddMenuItem(Consts.SHARP_BUILDER_TOOLS_VIEW_MENU_NAME, OTAMenuItemLocation.otamlChild, "ClipboardViewerMenu","Clipboard Viewer...");
			_MenuItem.Enabled = true;
			_MenuItem.Executed += new EventHandler(DoOpenClipboardViewer);

			// Sep2
			_MenuItem = _MenuService.AddMenuItem(Consts.SHARP_BUILDER_TOOLS_VIEW_MENU_NAME, OTAMenuItemLocation.otamlChild, "SepMenuEnhancement2","-");

			aMenuNames = new string[] {"ClipboardViewerMenu"};
			aMenuShortCuts = new int[] {0};

			fClipboardHook = new ClipboardHook();
			fClipboardHook.Changed += new EventHandler(ClipBoardChanged);
			fClipboardHook.Create();
		}
	}
}
